% Copyright 2014 Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000 with Sandia
% Corporation, the U.S. Government retains certain rights in this software
%
% This file is part of Sandia SPT (Sandia Simple Particle Tracking) v. 1.0.
% 
% Sandia SPT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Sandia SPT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Sandia SPT.  If not, see <http://www.gnu.org/licenses/>.

function [trajectories,rhoEstimate] = f2fTracking(positions,...
	maxDisplacement,directory,display)

%function [trajectories,rhoEstimate] = f2fTracking(positions,...
% 	maxDisplacement,directory,display)
%
%PURPOSE:
%	f2fTracking performs frame-to-frame tracking for all frames of the
%	movie, using the previously computed observation positions. The code
%	attempts to minimize the total displacement between the observations in
%	the pairs of frames, first ruling out any displacements that are above
%	a threshold. 
%
%DEPENDENCIES:
%	calcDisplacement.m
%	callLapjv.m
%		lapjv.m
%
%INPUTS:
%	positions:
%		Each row corresponds to a signal detected in a frame. The first
%		two columns correspond to the x and y coordinates respectively 
%		(in units of pixels). The third column corresponds to the frame
%		number in which that signal was detected. 
%	maxDisplacement:
%		A threshold value, the maximum allowed displacement (in units of
%		pixels) for a particle from one frame to the next. 
%	directory:
%		The directory to be analyzed, which should contain one file per
%		frame, named such that they appear sequentially in the directory. 
%	display:
%		A logical (true or false) value, indicating whether the code should
%		pause after each frame (true) displaying a grayscale image of the
%		frame with the localizations of the detected signals indicated by
%		superimposed red circles. 
%
%OUTPUTS:
%	trajectories:
%		Each row corresponds to a signal detected in a frame. The first
%		two columns correspond to the x and y coordinates respectively 
%		(in units of pixels). The third column corresponds to the frame
%		number in which that signal was detected. The fourth column
%		corresponds to the ID of the trajectory to which this observation
%		belongs. Each trajectory has a unique ID, with the exception of
%		those entries for which this value is 0. Rows whose fourth column
%		values are 0 correspond to observations which do not belong to any
%		trajectory, but are isolated points. 
%	rhoEstimate:
%		An estimate of the value of rho, where rho is the ratio of the
%		average motion of the particles in each frame (average
%		frame-to-frame displacement) to the average separation between
%		particles (average nearest neighbor distance). A useful guideline 
%		is that for typical particle tracking algorithms to perform well, 
%		rho should be well below 0.5. The value reported here is simply an
%		estimate of rho, but is often helpful in determining whether the
%		data is conducive to tracking. 
%
%LIMITATIONS:
%	1. As rho approaches 0.5, the tracking results will be unreliable. 
%	2. When there are observations in one frame which do not have
%	corresponding observations in the next frame, and vice versa, the
%	mathematically optimal pairing may result in a pairing which maximizes
%	the number of pairs formed at the expense of high average displacement
%	per pair. That is to say, instead of forming N pairs where each pair
%	has a very short displacement as a human might, the algorithm may form
%	N+1 pairs where nearest neighbors are rarely connected in order to
%	allow the generation of the additional pair. 
%	3. The mathematically optimal match may not always correspond to the
%	real physical trajectory. If the paths of two particles cross each
%	other, the mathematically optimal solution is to assume that the paths
%	did not cross. 
%
%NOTES:
%	Expected computational scaling is O(M*N^3) where M is the number of
%	frames and N is the number of positions per frame. 
%
%Written by Stephen M. Anthony 08/20/2014
%Last modified on 08/20/14 by Stephen M. Anthony


if exist('display','var') && display
	%Determines all files in the directory
	files = dir(directory);

	%Removes the first two entries, which are always included and 
	%correspond to the present directory and parent directory
	files = files(3:end);
end


%Determine the number of frames
nFrames = max(positions(:,3));

%Initialize variables for determining the reliability of the tracking
nearNeighbor = cell(nFrames,1);
displacementF2F = cell(nFrames,1);

%Initialize a count of the last used trajectory number
idCount = 0;

%Initialize the output. 
trajectories = [positions zeros(size(positions,1),1)];

%Code for progress display
digits = ceil(log10(nFrames));
numberFormat = ['%0' num2str(digits) 'd'];
backspace = {'\b'};
undo = cell2mat(backspace(ones(1,digits+1)));
fprintf(['There are ' num2str(nFrames) ' frames to process.\n']);
fprintf(['Currently working on frame #' num2str(1,numberFormat)]);


%Loop over all frame pairs
for frame = 1:(nFrames-1)
	%Display the progress. 
	fprintf([undo numberFormat '.'],frame);

	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	%Extracting the information from the relevant frames
	
	%Determine the row indices of the positions in the frame being examined
	%and the subsequent frame. 
	sources = positions(:,3)==frame;
	sinks = positions(:,3)==frame+1;
	
	%Extract the xy-coordinates for just the relevant positions
	sourcePos = positions(sources,1:2);
	sinkPos = positions(sinks,1:2);
	
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	%Calculations done to evaluate the reliability of the tracking, or
	%whether the combination of sample density and rate of motion are such
	%that tracking should not be trusted. 
	
	%Calculate the separation between all pairs of particles in the current
	%frame. 
	distSingleFrame = calcDisplacement(sourcePos,sourcePos);
	
	%Add an offset such that min will find the distance to the nearest
	%neighbor, not the distance from the observation to itself (which is
	%zero). The value used, 1e10, is large enough that no camera should
	%have a dimension with near that many pixels. 
	distSingleFrame = distSingleFrame  + eye(size(distSingleFrame))*1e10;
	nearNeighbor{frame} = min(distSingleFrame);
	
	%Calculate the displacement from each observation in one frame to
	%each observation in the next frame. 
	displacements = calcDisplacement(sourcePos,sinkPos);
	
	%Record the distance from each observation in one frame to the closest 
	%observation in the next frame. 
	displacementF2F{frame} = min(displacements);
	
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	%Determining the best matching between observations in one frame and
	%the next. 
	
	%Exclude any displacements which are above the threshold. 
	displacements(displacements> maxDisplacement) = NaN;
	
	%Find the solution to the linear sum assignment problem, determining
	%the best matching between the two sets of observations. 
	pairs = callLapjv(displacements);
	
	%When no alternative match is available, the algorithm will create
	%matches between particles which are separated by more than the maximum
	%displacement. Discard these matches. 
	lindex = sub2ind(size(displacements),pairs(:,1),pairs(:,2));
	keep = ~isnan(displacements(lindex));
	pairs = pairs(keep,:);
	
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	%Record the tracking
	
	%Determine whether any of the observations from the source frame which
	%were tracked have previously assigned trajectory numbers
	sourceIDs = trajectories(sources,4);
	
	%Determine which have not been assigned yet (and will be 0)
	unassigned = ~sourceIDs(pairs(:,1));
	nNew = sum(unassigned);
	
	%Assign trajectory ids, after which point all observations from the
	%source frame which are paired with an observation in the sink frame
	%will have uniquely assigned trajectory ids. 
	sourceIDs(pairs(unassigned,1)) = idCount + (1:nNew);
	idCount = idCount + nNew;
	
	%Record these in the trajectories variable. 
	trajectories(sources,4) = sourceIDs;
	
	%The sink frame will not yet have any trajectory ids (other than the
	%initialization value of 0). However, extract these so that the code
	%for the sinks parallels the code for the sources as much as possible. 
	sinkIDs = trajectories(sinks,4);
	
	%Foe each observation in the sink frame paired with an observation in
	%the source frame, propogate the trajectory id from the source frame to
	%the sink frame. 
	sinkIDs(pairs(:,2)) = sourceIDs(pairs(:,1));
	
	%Record these in the trajectories variable. 
	trajectories(sinks,4) = sinkIDs;
	
	
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	%Display the tracking if desired
	if exist('display','var') && display
		%Except for the pause, everything in this clause is a repeat of 
		%visualizeF2Ftrack
		image = imread(fullfile(directory,files(frame+1).name));
		imagesc(image); set(gca,'DataAspectRatio',[1 1 1])
		colormap('gray');
		hold on
		plot(sourcePos(:,1),sourcePos(:,2),'rx','LineWidth',2);
		plot(sinkPos(:,1),sinkPos(:,2),'bo','LineWidth',2);
		beg = sourcePos(pairs(:,1),:);
		fin = sinkPos(pairs(:,2),:);
		plot([beg(:,1) fin(:,1)]',[beg(:,2) fin(:,2)]','g','LineWidth',2);
		hold off
		pause
	end
end

%Display done
fprintf('\n');
disp('Done');

%Compute more accurate representations of these variables by using results
%from not just a single frame. 
estAvgNearNeighbor = mean(cell2mat(nearNeighbor'));
estAvgDisplacementF2F = mean(cell2mat(displacementF2F'));

%Estimate the value of rho, a measure of whether the tracking is reliable.
%Tracking works best when rho is well below 0.5. 
rhoEstimate = estAvgDisplacementF2F/ estAvgNearNeighbor;

if rhoEstimate>.25
	%Note, results below the threshold of 0.25 may also be unreliable.
	%However, there is a high enough probability of unreliable results when
	%rhoEstimate>.25 that issuing a warning seemed desirable. 
	warning('f2fTracking:Unreliable',...
		['The ratio of average frame-to-frame displacement to average '...
		'nearest neighbor separation is higher than desireable. The '...
		'tracking results may not be reliable'])
end

